<?php
require_once __DIR__ . '/Database.php';
require_once __DIR__ . '/Logger.php';

class AdminController
{
    private $pdo;
    private $logger;
    private $config;

    public function __construct()
    {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        $this->pdo = Database::connect();
        $this->logger = Logger::getInstance();
        $this->config = require __DIR__ . '/../config/config.php';
    }

    public function login()
    {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $username = $_POST['username'] ?? '';
            $password = $_POST['password'] ?? '';

            // Debug logging
            error_log("Login attempt - Username: $username");

            $stmt = $this->pdo->prepare("SELECT * FROM admins WHERE username = ?");
            $stmt->execute([$username]);
            $admin = $stmt->fetch();

            error_log("Admin found: " . ($admin ? "YES" : "NO"));

            if ($admin) {
                error_log("Password hash from DB: " . substr($admin['password_hash'], 0, 30));
                error_log("Verifying password...");
                $isValid = password_verify($password, $admin['password_hash']);
                error_log("Password verification result: " . ($isValid ? "VALID" : "INVALID"));
            }

            if ($admin && password_verify($password, $admin['password_hash'])) {
                error_log("Login successful for: $username");
                $_SESSION['admin_logged_in'] = true;
                $_SESSION['admin_id'] = $admin['id'];
                $_SESSION['admin_username'] = $admin['username'];
                $this->logAudit($admin['id'], 'LOGIN', 'Admin logged in');
                header('Location: admin.php?action=dashboard');
                exit;
            } else {
                error_log("Login failed - Invalid credentials");
                $this->logger->warning("Failed login attempt", ['username' => $username, 'ip' => $_SERVER['REMOTE_ADDR']]);
                $error = "Invalid credentials";
                require __DIR__ . '/../views/admin/login.php';
                return;
            }
        }
        require __DIR__ . '/../views/admin/login.php';
    }

    public function logout()
    {
        $adminId = $_SESSION['admin_id'] ?? null;
        if ($adminId) {
            $this->logAudit($adminId, 'LOGOUT', 'Admin logged out');
        }
        session_destroy();
        header('Location: admin.php?action=login');
        exit;
    }

    public function dashboard()
    {
        $this->requireAuth();

        // Time Ranges
        $range7d = "created_at > NOW() - INTERVAL 7 DAY";
        $range30d = "created_at > NOW() - INTERVAL 30 DAY";

        // KPI: Unique Visitors (Session based)
        $unique_visits_7d = $this->pdo->query("SELECT COUNT(DISTINCT session_uuid) FROM page_views WHERE $range7d")->fetchColumn() ?: 0;
        $unique_visits_30d = $this->pdo->query("SELECT COUNT(DISTINCT session_uuid) FROM page_views WHERE $range30d")->fetchColumn() ?: 0;

        // KPI: Total Leads
        $leads_7d = $this->pdo->query("SELECT COUNT(*) FROM leads WHERE $range7d")->fetchColumn() ?: 0;
        $leads_30d = $this->pdo->query("SELECT COUNT(*) FROM leads WHERE $range30d")->fetchColumn() ?: 0;
        $total_leads = $this->pdo->query("SELECT COUNT(*) FROM leads")->fetchColumn() ?: 0;

        // KPI: Conversion Rate
        $cr_7d = ($unique_visits_7d > 0) ? round(($leads_7d / $unique_visits_7d) * 100, 2) : 0;
        $cr_30d = ($unique_visits_30d > 0) ? round(($leads_30d / $unique_visits_30d) * 100, 2) : 0;

        // Breakdown: Top Countries
        $top_countries = $this->pdo->query("SELECT country_of_living, COUNT(*) as c FROM leads GROUP BY country_of_living ORDER BY c DESC LIMIT 5")->fetchAll() ?: [];

        // Create stats array for template
        $stats = [
            'total_leads' => $total_leads,
            'leads_7d' => $leads_7d,
            'unique_visits_30d' => $unique_visits_30d,
            'cr_30d' => $cr_30d
        ];

        // Recent Registrations
        $latest_leads = $this->pdo->query("SELECT * FROM leads ORDER BY created_at DESC LIMIT 10")->fetchAll() ?: [];

        require __DIR__ . '/../views/admin/dashboard.php';
    }

    public function registrations()
    {
        $this->requireAuth();

        $page = $_GET['page'] ?? 1;
        $limit = 20;
        $offset = ($page - 1) * $limit;

        $where = [];
        $params = [];

        // Search
        if (!empty($_GET['search'])) {
            $term = $_GET['search'];
            $where[] = "(full_name LIKE ? OR email LIKE ? OR whatsapp_e164 LIKE ?)";
            $params[] = "%$term%";
            $params[] = "%$term%";
            $params[] = "%$term%";
        }

        // Filters
        if (!empty($_GET['country'])) {
            $where[] = "country_of_living = ?";
            $params[] = $_GET['country'];
        }
        if (!empty($_GET['degree'])) {
            $where[] = "last_degree = ?";
            $params[] = $_GET['degree'];
        }
        if (!empty($_GET['date_from'])) {
            $where[] = "created_at >= ?";
            $params[] = $_GET['date_from'] . ' 00:00:00';
        }
        if (!empty($_GET['date_to'])) {
            $where[] = "created_at <= ?";
            $params[] = $_GET['date_to'] . ' 23:59:59';
        }

        $sql = "SELECT * FROM leads";
        if ($where) {
            $sql .= " WHERE " . implode(' AND ', $where);
        }

        // Count for pagination
        $countSql = str_replace("SELECT *", "SELECT COUNT(*)", $sql);
        $countStmt = $this->pdo->prepare($countSql);
        $countStmt->execute($params);
        $total_records = $countStmt->fetchColumn();
        $total_pages = ceil($total_records / $limit);

        // Fetch Data
        $sql .= " ORDER BY created_at DESC LIMIT $limit OFFSET $offset";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $leads = $stmt->fetchAll();

        // Dropdown Data (for filters)
        $countries = $this->pdo->query("SELECT DISTINCT country_of_living FROM leads ORDER BY country_of_living")->fetchAll(PDO::FETCH_COLUMN);
        $degrees = $this->pdo->query("SELECT DISTINCT last_degree FROM leads ORDER BY last_degree")->fetchAll(PDO::FETCH_COLUMN);

        require __DIR__ . '/../views/admin/registrations.php';
    }

    public function visits()
    {
        $this->requireAuth();

        $page = $_GET['page'] ?? 1;
        $limit = 50;
        $offset = ($page - 1) * $limit;

        $where = [];
        $params = [];

        // Filters
        if (!empty($_GET['source'])) {
            $where[] = "utm_source = ?";
            $params[] = $_GET['source'];
        }

        $sql = "SELECT * FROM page_views";
        if ($where) {
            $sql .= " WHERE " . implode(' AND ', $where);
        }

        // Count
        $countSql = str_replace("SELECT *", "SELECT COUNT(*)", $sql);
        $stmt = $this->pdo->prepare($countSql);
        $stmt->execute($params);
        $total_records = $stmt->fetchColumn();
        $total_pages = ceil($total_records / $limit);

        // Fetch
        $sql .= " ORDER BY created_at DESC LIMIT $limit OFFSET $offset";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $visits = $stmt->fetchAll();

        require __DIR__ . '/../views/admin/visits.php';
    }

    public function export_csv()
    {
        $this->requireAuth();

        $adminId = $_SESSION['admin_id'];

        // Apply filters if provided
        $where = [];
        $params = [];

        if (!empty($_GET['country'])) {
            $where[] = "country_of_living = ?";
            $params[] = $_GET['country'];
        }
        if (!empty($_GET['verified'])) {
            $where[] = "email_verified_at IS NOT NULL";
        }
        if (!empty($_GET['marketing'])) {
            $where[] = "marketing_consent = 1";
        }

        $sql = "SELECT id, full_name, email, whatsapp_e164, university, last_degree, country_of_living, ip_address, marketing_consent, email_verified_at, created_at FROM leads";

        if (!empty($where)) {
            $sql .= " WHERE " . implode(' AND ', $where);
        }

        $sql .= " ORDER BY created_at DESC";

        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            $records = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // Log the export action
            $this->logAudit($adminId, 'EXPORT_CSV', 'Exported ' . count($records) . ' records');

            // Set headers
            header('Content-Type: text/csv; charset=utf-8');
            header('Content-Disposition: attachment; filename="leads_export_' . date('Y-m-d_H-i-s') . '.csv"');

            $output = fopen('php://output', 'w');

            // BOM for Excel UTF-8 compatibility
            fprintf($output, chr(0xEF) . chr(0xBB) . chr(0xBF));

            // Write header row
            fputcsv($output, [
                'ID',
                'Full Name',
                'Email',
                'WhatsApp',
                'University',
                'Degree',
                'Country',
                'IP Address',
                'Marketing Consent',
                'Email Verified',
                'Registration Date'
            ]);

            // Write data rows with proper escaping
            foreach ($records as $row) {
                fputcsv($output, [
                    $row['id'],
                    $row['full_name'],
                    $row['email'],
                    $row['whatsapp_e164'],
                    $row['university'],
                    $row['last_degree'],
                    $row['country_of_living'],
                    $row['ip_address'],
                    $row['marketing_consent'] ? 'Yes' : 'No',
                    $row['email_verified_at'] ? 'Yes (' . $row['email_verified_at'] . ')' : 'No',
                    $row['created_at']
                ]);
            }

            fclose($output);
            exit;

        } catch (Exception $e) {
            $this->logger->exception($e, 'admin');
            die('Export failed: ' . $e->getMessage());
        }
    }

    public function exportDatabase()
    {
        $this->requireAuth();

        try {
            $timestamp = date('Y-m-d_H-i-s');
            $filename = "database_backup_{$timestamp}.sql";

            header('Content-Type: application/sql');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Cache-Control: no-cache, no-store, must-revalidate');
            header('Pragma: no-cache');
            header('Expires: 0');

            $output = fopen('php://output', 'w');

            // Write SQL header
            fwrite($output, "-- Database Backup\n");
            fwrite($output, "-- Generated: " . date('Y-m-d H:i:s') . "\n");
            fwrite($output, "-- Database: " . $this->config['db']['dbname'] . "\n\n");

            // Get all tables
            $stmt = $this->pdo->prepare("SELECT TABLE_NAME FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = ?");
            $stmt->execute([$this->config['db']['dbname']]);
            $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);

            foreach ($tables as $table) {
                // Drop table if exists
                fwrite($output, "\nDROP TABLE IF EXISTS `$table`;\n");

                // Get table structure
                $stmt = $this->pdo->prepare("SHOW CREATE TABLE `$table`");
                $stmt->execute();
                $createTable = $stmt->fetch(PDO::FETCH_ASSOC);
                fwrite($output, $createTable['Create Table'] . ";\n\n");

                // Get table data
                $stmt = $this->pdo->prepare("SELECT * FROM `$table`");
                $stmt->execute();
                $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

                if (!empty($rows)) {
                    // Get column names
                    $columns = array_keys($rows[0]);
                    $columnNames = implode('`, `', $columns);

                    foreach ($rows as $row) {
                        $values = array_map(function($value) {
                            if ($value === null) {
                                return 'NULL';
                            }
                            return "'" . addslashes($value) . "'";
                        }, array_values($row));

                        $valueString = implode(', ', $values);
                        fwrite($output, "INSERT INTO `$table` (`$columnNames`) VALUES ($valueString);\n");
                    }
                    fwrite($output, "\n");
                }
            }

            // Log the export
            $this->logAudit($_SESSION['admin_id'], 'EXPORT_DATABASE', 'Database exported to SQL file');

            fclose($output);
            exit;

        } catch (Exception $e) {
            $this->logger->exception($e, 'admin');
            die('Database export failed: ' . $e->getMessage());
        }
    }

    private function requireAuth()
    {
        if (empty($_SESSION['admin_logged_in'])) {
            header('Location: admin.php?action=login');
            exit;
        }
    }

    private function logAudit(int $adminId, string $action, string $description = '')
    {
        try {
            $ip = $_SERVER['REMOTE_ADDR'];
            $ua = $_SERVER['HTTP_USER_AGENT'] ?? '';

            $stmt = $this->pdo->prepare("INSERT INTO admin_audit_log (admin_id, action, description, ip_address, user_agent) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([$adminId, $action, $description, $ip, $ua]);
        } catch (Exception $e) {
            $this->logger->exception($e, 'admin');
        }
    }
}
