<?php

require_once __DIR__ . '/Database.php';
require_once __DIR__ . '/LeadService.php';
require_once __DIR__ . '/SecurityHelper.php';
require_once __DIR__ . '/Logger.php';

class LeadController
{
    private $leadService;
    private $logger;
    private $config;

    public function __construct()
    {
        $this->leadService = new LeadService();
        $this->logger = Logger::getInstance();
        $this->config = require __DIR__ . '/../config/config.php';
    }

    public function submit()
    {
        header('Content-Type: application/json');

        // 1. Request Method Validation
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            echo json_encode(['success' => false, 'message' => 'Method not allowed.']);
            return;
        }

        // 2. CSRF Token Validation
        $csrfToken = $_POST['csrf_token'] ?? '';
        if (!SecurityHelper::validateCSRFToken($csrfToken)) {
            http_response_code(403);
            echo json_encode(['success' => false, 'message' => 'Security token validation failed.']);
            $this->logger->warning("CSRF token validation failed", ['ip' => $_SERVER['REMOTE_ADDR']]);
            return;
        }

        // 3. Honeypot check
        if (!empty($_POST['hp_field'])) {
            echo json_encode(['success' => true, 'message' => 'Submission received.']);
            $this->logger->info("Bot detected via honeypot", ['ip' => $_SERVER['REMOTE_ADDR']]);
            return;
        }

        // 4. Input Sanitization & Validation
        $email = filter_input(INPUT_POST, 'email', FILTER_VALIDATE_EMAIL);
        $fullName = $_POST['full_name'] ?? '';
        $whatsapp = $_POST['whatsapp_e164'] ?? '';
        $university = $_POST['university'] ?? '';
        $degree = $_POST['last_degree'] ?? '';
        $country = $_POST['country_of_living'] ?? '';
        $marketingConsent = isset($_POST['marketing_consent']) ? 1 : 0;
        $termsConsent = isset($_POST['terms_consent']) ? 1 : 0;

        // Validate all fields
        if (!$email || !SecurityHelper::validateEmail($email)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Invalid email address.']);
            return;
        }

        if (!$fullName || !SecurityHelper::validateName($fullName)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Invalid full name.']);
            return;
        }

        if (!$whatsapp || !SecurityHelper::validateWhatsApp($whatsapp)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Invalid WhatsApp number (must be in E.164 format, e.g., +12345678900).']);
            return;
        }

        if (!SecurityHelper::validateUniversity($university)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Invalid university name.']);
            return;
        }

        if (!SecurityHelper::validateDegree($degree)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Invalid degree selection.']);
            return;
        }

        if (!$country || !SecurityHelper::validateCountry($country)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Invalid country name.']);
            return;
        }

        if (!$termsConsent) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'You must agree to the Terms & Privacy Policy.']);
            return;
        }

        // 5. Rate Limiting (IP based)
        $ip = $this->getClientIP();
        $rateLimit = $this->leadService->checkRateLimit($ip, 5, 10);

        if ($rateLimit['limited']) {
            http_response_code(429);
            echo json_encode([
                'success' => false,
                'message' => 'Too many attempts. Please try again in a few minutes.'
            ]);
            $this->logger->warning("Rate limit exceeded", ['ip' => $ip, 'attempts' => $rateLimit['attempts']]);
            return;
        }

        // 6. Create Lead
        $leadData = [
            'full_name' => $fullName,
            'whatsapp_e164' => $whatsapp,
            'email' => $email,
            'university' => $university,
            'last_degree' => $degree,
            'country_of_living' => $country,
            'ip' => $ip,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown',
            'marketing_consent' => $marketingConsent,
            'terms_consent' => $termsConsent,
        ];

        $result = $this->leadService->createLead($leadData);

        if (!$result['success']) {
            http_response_code($result['isDuplicate'] ?? false ? 409 : 500);
            echo json_encode(['success' => false, 'message' => $result['message']]);
            return;
        }

        // 7. Generate Download Link (valid for 1 hour, no email sent)
        $downloadLink = $this->config['app']['base_url'] . "/download?token=" . $result['token'];

        echo json_encode([
            'success' => true,
            'message' => 'Registration successful! Your download link is ready.',
            'download_link' => $downloadLink
        ]);
    }

    public function download()
    {
        $token = $_GET['token'] ?? '';

        if (!$token || !is_string($token) || strlen($token) !== 64) {
            $this->renderError("Invalid download token.");
            return;
        }

        // Validate token
        $lead = $this->leadService->findByToken($token);

        if (!$lead) {
            $this->renderError("Invalid or expired download link. Please register again.");
            return;
        }

        // Mark email as verified
        $this->leadService->markEmailVerified($lead['id']);

        // Serve File
        $filePath = __DIR__ . '/../assets/book.pdf';

        if (!file_exists($filePath)) {
            // Create a dummy PDF for testing if it doesn't exist
            header("Content-type: application/pdf");
            header("Content-Disposition: inline; filename=ovarian_stimulation.pdf");
            echo "%PDF-1.4\n1 0 obj\n<< /Title (Ovarian Stimulation) /Creator (System) >>\nendobj\ntrailer\n<< /Root 1 0 R >>";
            exit;
        }

        header('Content-Description: File Transfer');
        header('Content-Type: application/pdf');
        header('Content-Disposition: attachment; filename="Ovarian_Stimulation_Rezk_Al-Inany.pdf"');
        header('Expires: 0');
        header('Cache-Control: must-revalidate');
        header('Pragma: public');
        header('Content-Length: ' . filesize($filePath));
        readfile($filePath);
        exit;
    }

    private function getClientIP()
    {
        $ip = $_SERVER['REMOTE_ADDR'];

        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
            $ip = trim($ips[0]);
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED'])) {
            $ip = $_SERVER['HTTP_X_FORWARDED'];
        } elseif (!empty($_SERVER['HTTP_FORWARDED_FOR'])) {
            $ip = $_SERVER['HTTP_FORWARDED_FOR'];
        } elseif (!empty($_SERVER['HTTP_FORWARDED'])) {
            $ip = $_SERVER['HTTP_FORWARDED'];
        }

        // Validate IP
        if (filter_var($ip, FILTER_VALIDATE_IP)) {
            return $ip;
        }

        return $_SERVER['REMOTE_ADDR'];
    }

    private function sendEmail($to, $name, $link)
    {
        $subject = "Your Download Link: Ovarian Stimulation in IVF";
        
        $htmlBody = $this->getEmailTemplate($name, $link);
        
        $headers = "MIME-Version: 1.0\r\n";
        $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
        $headers .= "From: " . $this->config['mail']['from_name'] . " <" . $this->config['mail']['from_email'] . ">\r\n";
        $headers .= "Reply-To: " . $this->config['mail']['from_email'] . "\r\n";
        $headers .= "X-Mailer: PHP/" . phpversion() . "\r\n";

        $sent = mail($to, $subject, $htmlBody, $headers);

        if ($sent) {
            $this->logger->info("Email sent", ['to' => $to, 'subject' => $subject]);
        } else {
            $this->logger->warning("Email failed to send", ['to' => $to, 'subject' => $subject]);
        }
    }

    private function getEmailTemplate($name, $link)
    {
        return <<<HTML
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Download Your Book</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Roboto', 'Helvetica', 'Arial', sans-serif;
            line-height: 1.6;
            color: #333;
            background-color: #f5f5f5;
        }
        .container {
            max-width: 600px;
            margin: 0 auto;
            background-color: #ffffff;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px 20px;
            text-align: center;
        }
        .header h1 {
            margin: 0;
            font-size: 24px;
            font-weight: 600;
        }
        .content {
            padding: 30px 20px;
        }
        .content p {
            margin: 15px 0;
            font-size: 16px;
        }
        .download-btn {
            display: inline-block;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 12px 30px;
            text-decoration: none;
            border-radius: 4px;
            font-weight: 600;
            margin: 20px 0;
            text-align: center;
        }
        .download-btn:hover {
            opacity: 0.95;
        }
        .link-text {
            font-size: 12px;
            color: #666;
            word-break: break-all;
            margin-top: 15px;
            padding: 10px;
            background-color: #f9f9f9;
            border-left: 3px solid #667eea;
        }
        .footer {
            background-color: #f9f9f9;
            padding: 20px;
            text-align: center;
            font-size: 12px;
            color: #666;
            border-top: 1px solid #eee;
        }
        .expiration {
            background-color: #fff3cd;
            border: 1px solid #ffc107;
            color: #856404;
            padding: 10px;
            border-radius: 4px;
            margin: 15px 0;
            font-size: 14px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>📚 Your Download is Ready</h1>
        </div>
        
        <div class="content">
            <p>Dear <strong>$name</strong>,</p>
            
            <p>Thank you for registering! We're excited to share the comprehensive guide on <strong>Ovarian Stimulation in IVF</strong> by Prof. Rezk & Prof. Al-Inany with you.</p>
            
            <p>Click the button below to download your copy:</p>
            
            <div style="text-align: center;">
                <a href="$link" class="download-btn">📥 Download PDF</a>
            </div>
            
            <div class="expiration">
                ⏰ <strong>Link expires in 2 hours</strong> from the time you registered. Please download it soon!
            </div>
            
            <p>If the button above doesn't work, you can copy and paste this link in your browser:</p>
            <div class="link-text">$link</div>
            
            <p>This is a secure, one-time download link. It will expire after 2 hours for security reasons.</p>
            
            <p>If you have any questions or didn't request this download, please feel free to contact us.</p>
            
            <p>Best regards,<br>
            The Ovarian Stimulation Team</p>
        </div>
        
        <div class="footer">
            <p>&copy; 2026 Ovarian Stimulation in IVF. All rights reserved.</p>
            <p>This is an automated email. Please do not reply directly to this message.</p>
        </div>
    </div>
</body>
</html>
HTML;
    }

    private function renderError($msg)
    {
        http_response_code(403);
        ?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Access Denied</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Roboto', sans-serif;
            display: flex;
            align-items: center;
            justify-content: center;
            min-height: 100vh;
            margin: 0;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        .container {
            background: white;
            padding: 40px;
            border-radius: 8px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.1);
            text-align: center;
            max-width: 500px;
        }
        h1 {
            color: #d32f2f;
            margin: 0 0 15px 0;
        }
        p {
            color: #666;
            line-height: 1.6;
            margin: 15px 0;
        }
        a {
            display: inline-block;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 10px 20px;
            text-decoration: none;
            border-radius: 4px;
            margin-top: 20px;
            font-weight: 600;
        }
        a:hover {
            opacity: 0.95;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>❌ Access Denied</h1>
        <p><?= htmlspecialchars($msg, ENT_QUOTES, 'UTF-8') ?></p>
        <a href="<?= htmlspecialchars($this->config['app']['base_url'], ENT_QUOTES, 'UTF-8') ?>">← Return to Home</a>
    </div>
</body>
</html>
        <?php
        exit;
    }
}
