<?php

require_once __DIR__ . '/Database.php';
require_once __DIR__ . '/Logger.php';

class LeadService
{
    private $pdo;
    private $logger;
    private $config;

    public function __construct()
    {
        $this->pdo = Database::connect();
        $this->logger = Logger::getInstance();
        $this->config = require __DIR__ . '/../config/config.php';
    }

    public function createLead(array $data)
    {
        try {
            // Check for duplicate email
            $duplicate = $this->findByEmail($data['email']);
            if ($duplicate) {
                return [
                    'success' => false,
                    'isDuplicate' => true,
                    'message' => 'This email is already registered. Please check your inbox for the download link or contact support.',
                    'existingLead' => $duplicate
                ];
            }

            // Generate download token
            $rawToken = bin2hex(random_bytes(32));
            $tokenSalt = $this->config['security']['token_salt'];
            $tokenHash = hash('sha256', $rawToken . $tokenSalt);
            $expiresAt = date('Y-m-d H:i:s', strtotime("+{$this->config['app']['download_validity_hours']} hours"));

            // Get geo data (placeholder - can integrate with IP geolocation service)
            $geoData = json_encode(['ip' => $data['ip'], 'note' => 'Geo lookup placeholder']);

            $sql = "INSERT INTO leads (full_name, whatsapp_e164, email, university, last_degree, country_of_living, ip_address, estimated_location, user_agent, marketing_consent, terms_consent, download_token_hash, download_expires_at) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

            $stmt = $this->pdo->prepare($sql);
            $result = $stmt->execute([
                $data['full_name'],
                $data['whatsapp_e164'],
                $data['email'],
                $data['university'],
                $data['last_degree'],
                $data['country_of_living'],
                $data['ip'],
                $geoData,
                $data['user_agent'],
                $data['marketing_consent'],
                $data['terms_consent'],
                $tokenHash,
                $expiresAt
            ]);

            if ($result) {
                $leadId = $this->pdo->lastInsertId();
                $this->logger->info("Lead created: {$data['email']}", ['lead_id' => $leadId, 'ip' => $data['ip']]);

                return [
                    'success' => true,
                    'lead_id' => $leadId,
                    'token' => $rawToken,
                    'message' => 'Lead created successfully'
                ];
            }

            throw new Exception('Failed to insert lead');

        } catch (Exception $e) {
            $this->logger->exception($e, 'leads');
            return [
                'success' => false,
                'message' => 'An error occurred. Please try again later.'
            ];
        }
    }

    public function findByEmail(string $email)
    {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM leads WHERE email = ? AND email_verified_at IS NOT NULL ORDER BY created_at DESC LIMIT 1");
            $stmt->execute([$email]);
            return $stmt->fetch();
        } catch (Exception $e) {
            $this->logger->exception($e, 'leads');
            return null;
        }
    }

    public function findByToken(string $token)
    {
        try {
            $tokenSalt = $this->config['security']['token_salt'];
            $tokenHash = hash('sha256', $token . $tokenSalt);

            $stmt = $this->pdo->prepare("SELECT * FROM leads WHERE download_token_hash = ? AND download_expires_at > NOW()");
            $stmt->execute([$tokenHash]);
            return $stmt->fetch();
        } catch (Exception $e) {
            $this->logger->exception($e, 'leads');
            return null;
        }
    }

    public function markEmailVerified(int $leadId)
    {
        try {
            $stmt = $this->pdo->prepare("UPDATE leads SET email_verified_at = NOW() WHERE id = ?");
            return $stmt->execute([$leadId]);
        } catch (Exception $e) {
            $this->logger->exception($e, 'leads');
            return false;
        }
    }

    public function checkRateLimit(string $ip, int $maxAttempts = 5, int $windowMinutes = 10)
    {
        try {
            $stmt = $this->pdo->prepare("SELECT COUNT(*) FROM leads WHERE ip_address = ? AND created_at > (NOW() - INTERVAL ? MINUTE)");
            $stmt->execute([$ip, $windowMinutes]);
            $count = $stmt->fetchColumn();

            return [
                'limited' => $count >= $maxAttempts,
                'attempts' => $count,
                'remaining' => max(0, $maxAttempts - $count)
            ];
        } catch (Exception $e) {
            $this->logger->exception($e, 'leads');
            return ['limited' => true, 'attempts' => 0, 'remaining' => 0];
        }
    }

    public function getAnalytics(string $range = '7d')
    {
        try {
            $rangeSQL = match ($range) {
                '7d' => "created_at > NOW() - INTERVAL 7 DAY",
                '30d' => "created_at > NOW() - INTERVAL 30 DAY",
                '90d' => "created_at > NOW() - INTERVAL 90 DAY",
                'all' => "1=1",
                default => "created_at > NOW() - INTERVAL 7 DAY"
            };

            $totalLeads = $this->pdo->query("SELECT COUNT(*) FROM leads WHERE $rangeSQL")->fetchColumn();
            $verifiedLeads = $this->pdo->query("SELECT COUNT(*) FROM leads WHERE $rangeSQL AND email_verified_at IS NOT NULL")->fetchColumn();

            return [
                'total' => $totalLeads,
                'verified' => $verifiedLeads,
                'conversion_rate' => $totalLeads > 0 ? round(($verifiedLeads / $totalLeads) * 100, 2) : 0
            ];
        } catch (Exception $e) {
            $this->logger->exception($e, 'leads');
            return ['total' => 0, 'verified' => 0, 'conversion_rate' => 0];
        }
    }

    public function getTopCountries(int $limit = 10)
    {
        try {
            $stmt = $this->pdo->query("SELECT country_of_living, COUNT(*) as count FROM leads GROUP BY country_of_living ORDER BY count DESC LIMIT $limit");
            return $stmt->fetchAll();
        } catch (Exception $e) {
            $this->logger->exception($e, 'leads');
            return [];
        }
    }

    public function getAllLeads(array $filters = [], int $limit = 20, int $offset = 0)
    {
        try {
            $where = [];
            $params = [];

            if (!empty($filters['search'])) {
                $term = "%{$filters['search']}%";
                $where[] = "(full_name LIKE ? OR email LIKE ? OR whatsapp_e164 LIKE ?)";
                $params = array_merge($params, [$term, $term, $term]);
            }

            if (!empty($filters['country'])) {
                $where[] = "country_of_living = ?";
                $params[] = $filters['country'];
            }

            if (!empty($filters['degree'])) {
                $where[] = "last_degree = ?";
                $params[] = $filters['degree'];
            }

            if (!empty($filters['date_from'])) {
                $where[] = "created_at >= ?";
                $params[] = $filters['date_from'] . ' 00:00:00';
            }

            if (!empty($filters['date_to'])) {
                $where[] = "created_at <= ?";
                $params[] = $filters['date_to'] . ' 23:59:59';
            }

            $whereClause = !empty($where) ? " WHERE " . implode(' AND ', $where) : "";

            $countStmt = $this->pdo->prepare("SELECT COUNT(*) FROM leads" . $whereClause);
            $countStmt->execute($params);
            $total = $countStmt->fetchColumn();

            $stmt = $this->pdo->prepare("SELECT * FROM leads" . $whereClause . " ORDER BY created_at DESC LIMIT ? OFFSET ?");
            $stmt->execute(array_merge($params, [$limit, $offset]));

            return [
                'leads' => $stmt->fetchAll(),
                'total' => $total,
                'page' => ceil($offset / $limit) + 1,
                'pages' => ceil($total / $limit)
            ];
        } catch (Exception $e) {
            $this->logger->exception($e, 'leads');
            return ['leads' => [], 'total' => 0, 'page' => 1, 'pages' => 0];
        }
    }
}
