<?php

class Logger
{
    private static $logDir;
    private static $instance;

    private function __construct()
    {
        self::$logDir = __DIR__ . '/../logs';
        if (!is_dir(self::$logDir)) {
            mkdir(self::$logDir, 0755, true);
        }
    }

    public static function getInstance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function error(string $message, array $context = [], string $channel = 'error')
    {
        $this->log('ERROR', $message, $context, $channel);
    }

    public function warning(string $message, array $context = [], string $channel = 'warning')
    {
        $this->log('WARNING', $message, $context, $channel);
    }

    public function info(string $message, array $context = [], string $channel = 'info')
    {
        $this->log('INFO', $message, $context, $channel);
    }

    public function exception(\Exception $e, string $channel = 'exception')
    {
        $context = [
            'code' => $e->getCode(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString(),
        ];
        $this->log('EXCEPTION', $e->getMessage(), $context, $channel);
    }

    private function log(string $level, string $message, array $context = [], string $channel = 'app')
    {
        $timestamp = date('Y-m-d H:i:s');
        $logFile = self::$logDir . '/' . $channel . '-' . date('Y-m-d') . '.log';

        $logMessage = "[$timestamp] [$level] $message";
        if (!empty($context)) {
            $logMessage .= "\nContext: " . json_encode($context, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        }
        $logMessage .= "\n" . str_repeat('-', 80) . "\n";

        error_log($logMessage, 3, $logFile);
    }

    public function getLogs(string $channel = 'error', int $days = 7)
    {
        $logs = [];
        $now = time();
        $dayInSeconds = 86400;

        for ($i = 0; $i < $days; $i++) {
            $date = date('Y-m-d', $now - ($i * $dayInSeconds));
            $logFile = self::$logDir . '/' . $channel . '-' . $date . '.log';

            if (file_exists($logFile)) {
                $logs[$date] = file_get_contents($logFile);
            }
        }

        return $logs;
    }
}
