<?php

class SecurityHelper
{
    private static $tokenLength = 32;

    public static function generateCSRFToken()
    {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }

        if (empty($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(self::$tokenLength));
        }

        return $_SESSION['csrf_token'];
    }

    public static function validateCSRFToken(string $token)
    {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }

        return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
    }

    public static function sanitizeText(string $input, int $maxLength = 255)
    {
        $input = trim($input);
        $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
        return substr($input, 0, $maxLength);
    }

    public static function validateName(string $name)
    {
        $name = self::sanitizeText($name, 255);
        if (strlen($name) < 2 || strlen($name) > 255) {
            return false;
        }
        // Allow letters, spaces, hyphens, and apostrophes
        return preg_match("/^[a-zA-Z\s\-']+$/u", $name) === 1;
    }

    public static function validateEmail(string $email)
    {
        return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
    }

    public static function validateWhatsApp(string $phone)
    {
        // E.164 format: +[country code][number] (7-15 digits)
        // More lenient: allows spaces, hyphens, parentheses, and strips them for validation
        $cleaned = preg_replace('/[\s\-\(\)]/', '', $phone);
        return preg_match('/^\+[1-9]\d{1,14}$/', $cleaned) === 1;
    }

    public static function validateCountry(string $country)
    {
        $country = self::sanitizeText($country, 100);
        if (strlen($country) < 2 || strlen($country) > 100) {
            return false;
        }
        return preg_match("/^[a-zA-Z\s\-']+$/u", $country) === 1;
    }

    public static function validateDegree(string $degree)
    {
        // Accept any reasonable text input for degree (2-100 chars)
        $degree = self::sanitizeText($degree, 100);
        return strlen($degree) >= 2 && strlen($degree) <= 100;
    }

    public static function validateUniversity(string $university)
    {
        $university = self::sanitizeText($university, 255);
        return strlen($university) >= 3 && strlen($university) <= 255;
    }

    public static function sanitizeForHTML(string $input)
    {
        return htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
    }

    public static function sanitizeForSQL(string $input)
    {
        // Not needed with parameterized queries, but available for reference
        return addslashes($input);
    }
}
